%%%% A Matlab Code for 3D Non-Penalization SEMDOT Algorithm %%%%
% Example SEMDOT3D(60,60,60,0.5,3.5)
function SEMDOT3D(nelx,nely,nelz,vol,rmin)
%% USER-DEFINED LOOP PARAMETERS
maxloop = 5000; tolx = 0.001; nele = nelx*nely*nelz;
%% USER-DEFINED MATERIAL PROPERTIES
E0 = 1; Emin = 0.001; nu = 0.3; tol = 1; k0 = 1; kmin = 0.001;
scale_obj = 0.001;
%% USER-DEFINED GRID POINTS
nG = 5; ngrid = nG-1;
%% INITIALIZE HEAVISIDE REGULARIZATION PARAMETER
beta = 0.5; ER = 0.5;
%% ELEMENTAL NODES AND COORDINATES
[nodex,nodey,nodez] = meshgrid(0:nelx,0:nely,0:nelz);
[fnx,fny,fnz] = meshgrid(0:1/ngrid:nelx,0:1/ngrid:nely,0:1/ngrid:nelz);
%% USER-DEFINED LOAD DOFs
il=nelx/2-nelx/20:nelx/2+nelx/20;
jl=0; kl=nelz/2-nelz/20:nelz/2+nelz/20;
fixedxy=il*(nely+1)+(nely+1-jl);
fixednid=repmat(fixedxy',size(kl))+repmat(kl*(nelx+1)*(nely+1),size(fixedxy,2),1);
fixeddof=reshape(fixednid,[],1);
%% PREPARE FINITE ELEMENT ANALYSIS
ndof = (nelx+1)*(nely+1)*(nelz+1);
F = sparse(1:ndof,1,-0.01,ndof,1);
U = zeros(ndof,1);
freedofs = setdiff(1:ndof,fixeddof);
KE = lk_H8(k0);
nodegrd = reshape(1:(nely+1)*(nelx+1),nely+1,nelx+1);
nodeids = reshape(nodegrd(1:end-1,1:end-1),nely*nelx,1);
nodeidz = 0:(nely+1)*(nelx+1):(nelz-1)*(nely+1)*(nelx+1);
nodeids = repmat(nodeids,size(nodeidz))+repmat(nodeidz,size(nodeids));
edofVec = nodeids(:)+1;
edofMat = repmat(edofVec,1,8)+ ...
    repmat([0 nely + [1 0] -1 ...
    (nely+1)*(nelx+1)+[0 nely + [1 0] -1]],nele,1);
iK = reshape(kron(edofMat,ones(8,1))',8*8*nele,1);
jK = reshape(kron(edofMat,ones(1,8))',8*8*nele,1);
%% FILTER
[H,Hs]=HHs(nelx,nely,nelz,rmin);
[Hn,Hns]=HnHns(nelx,nely,nelz);
%% INITIALIZE ITERATION
vx = repmat(vol,[nely,nelx,nelz]); vxPhys = vx; loop = 0; change = 1;
%% INITIALIZE MMA OPTIMIZER
m = 1; n = nele; xmin = zeros(n,1); xmax = ones(n,1); xold1 = vx(:);            
xold2 = vx(:); low = ones(n,1); upp = ones(n,1); a0 = 1; a = zeros(m,1);       
c_MMA = 10000*ones(m,1); d = zeros(m,1);       
%% START ITERATION
while (change > tolx || tol>0.001) && loop < maxloop
    %% UPDATE ITERATION
    loop = loop+1;
    %% FE-ANALYSIS
    sK = reshape(KE(:)*(vxPhys(:)'*E0+(1-vxPhys(:))'*(Emin*E0)),8*8*nele,1);
    K = sparse(iK(:),jK(:),sK(:)); K = (K+K')/2;
    U(freedofs,:) = K(freedofs, freedofs)\F(freedofs,:);
    %% OBJECTIVE FUNCTION AND SENSITIVITY ANALYSIS
    ce = reshape(sum((U(edofMat)*KE).*U(edofMat),2),[nely,nelx,nelz]);
    c(loop) = sum(sum(sum((vxPhys.*E0+(1-vxPhys).*(Emin*E0)).*ce)));
    dc = -((1-vxPhys)*Emin+vxPhys).*E0.*ce;
    dv = ones(nely,nelx,nelz);
    %% FILTERING AND MODIFICATION OF SENSITIVITIES
    dc(:) = H*(dc(:)./Hs);
    dv(:) = H*(dv(:)./Hs);
    %% METHOD OF MOVING ASYMPTOTES
    xval = vx(:);
    f0val = scale_obj*c;
    df0dx = scale_obj*dc(:);
    fval = sum(vxPhys(:))/(vol*nele) - 1;
    dfdx = dv(:)' / (vol*nele);
    [vxmma, ~, ~, ~, ~, ~, ~, ~, ~, low,upp] = ...
        mmasub(m, n, loop, xval, xmin, xmax, xold1, xold2, ...
        f0val,df0dx,fval,dfdx,low,upp,a0,a,c_MMA,d);
    %% Update MMA Variables
    vxnew = reshape(vxmma,nely,nelx,nelz);
    vxPhys(:) = (H*vxnew(:))./Hs;
    xold2 = xold1(:);
    xold1 = vx(:);
    xn = reshape((Hn*vxPhys(:)./Hns),nely+1,nelx+1,nelz+1);
    %% UPDATE POINT DESNIGY BY A HEAVISIDE SMOOTH FUNCTION/ HEAVISIDE STEP FUNCTION
    xg = interp3(nodex,nodey,nodez,xn,fnx,fny,fnz,'linear');
    l1 =0; l2 = 1;
    while (l2-l1) > 1.0e-5
        ls = (l1+l2)/2.0;
        %% Heaviside smooth function
        xgnew = max(0.001,(tanh(beta*ls)+tanh(beta*(xg-ls)))/(tanh(beta*ls)+tanh(beta*(1-ls))));
        if sum(sum(sum(xgnew)))/((ngrid*nelx+1)*(ngrid*nely+1)*(ngrid*nelz+1)) - sum(vxPhys(:))/(nele) > 0
            l1 = ls;
        else
            l2 = ls;
        end
    end
    %% CONVERTING TO ELEMENTS
    vxPhys(:) = 0;
    Terr = 0;
    Tm=[];
    for nk = 1:nelz
        for ni = 1:nelx
            for nj = 1:nely
                ne = (nk-1)*nelx*nely+(ni-1)*nely+nj;
                for nk1 = ngrid*(nk-1)+1:ngrid*nk+1
                    for ni1 = ngrid*(ni-1)+1:ngrid*ni+1
                        for nj1 = ngrid*(nj-1)+1:ngrid*nj+1
                            Tm=[Tm;xgnew(nj1,ni1,nk1)];
                            vxPhys(ne) = vxPhys(ne)+xgnew(nj1,ni1,nk1);
                        end
                    end
                end
                if min(Tm)>0.001 && max(Tm)<1
                    Terr=Terr+1;
                end
                Tm=[];
            end
        end
    end
    vxPhys = vxPhys/((ngrid+1)^3);
    %% Topology Error
    tol = Terr/nele;
    %% CHECK CONVERGENCE
    change = sum(abs(vxnew(:)-vx(:)))/(vol*nele);
    vx=vxnew;
    %% PRINT RESULTS
    fprintf('It.:%3i Obj.:%8.4f Vol.:%4.3f ch.:%4.5f Topo.:%7.5f Beta.:%g.\n',...
        loop,c(loop),mean(vxPhys(:)),change,tol,beta);
    clf;
    top = xg-ls;
    display_3Dsmooth(xg,top,loop);
    axis equal tight off; drawnow
    %% UPDATE% HEAVISIDE REGULARIZATION PARAMETER
    beta=beta+ER;
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This code was written by:                                               %
%                                                                         %
% Dr. Yun-Fei Fu                                                          %
% Department of Mechanical Engineering, University of Alberta, Canada     %
%                                                                         %
% Prof. Bernard Rolfe                                                     %
% School of Engineering, Faculty of Science Engineering and               %
% Built Environment, Deakin University, Australia                         %
%                                                                         %
% Please send your comments and questions to: jackyunfei.fu@outlook.com   %
%                                                                         %
% The code is intended for educational purposes and theoretical details   %
% are discussed in the book: "Smooth Topological Design of Continuum      %
% Structures", YF Fu and B Rolfe, CRC press.                              %
%                                                                         %
%                                                                         %
% The basic structure of the code is based on the 3D code from:           %
% "An efficient 3D topology optimization code written in Matlab",         %
% K Liu and A Tovar, SMO 2014, doi:10.1007/s00158-014-1107-x              %
%                                                                         %
% Disclaimer:                                                             %
% The authors do not guarantee that the code is free from errors.         %
% Furthermore, the authors shall not be liable in any event caused by the %
% use of the program.                                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%